import uuid
from django.db import models
from django.contrib.auth.models import User
from advertising.models import ads
from django.utils import timezone
from backend.validators import validate_file_extension
from django.core.validators import FileExtensionValidator

from PIL import Image, ExifTags

class Page(models.Model):
    pageid = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    pagename = models.CharField(max_length=200)
    advertising = models.ForeignKey(ads, on_delete=models.CASCADE, parent_link=True, null=True)
    views = models.IntegerField(default=0)
    shares = models.IntegerField(default=0)
    pub_date = models.DateTimeField(auto_now_add=True)
    edit_date = models.DateTimeField(auto_now=True)
    user = models.ForeignKey(User, on_delete=models.CASCADE, parent_link=True)

    def __str__(self):
        return self.pageid

    def __repr__(self):
        return str(self.pageid.hex)

    def increment_views(self):
        self.views += 1
        self.save()

    def name(self):
        return self.name

    def viewsformat(self):
        if self.views > 1000:
            views = "{0:0.1f}K".format(self.views/1000)
            return views
        else:
            return self.views


class Pageredirects(models.Model):
    page = models.ForeignKey(Page, on_delete=models.CASCADE)
    user = models.ForeignKey(User, on_delete=models.CASCADE, parent_link=True)
    redirectname = models.CharField(max_length=44, primary_key=True)
    customname = models.CharField(max_length=30)

class Themes(models.Model):
    classname = models.CharField(max_length=30, blank=True)
    allowed_users = models.ManyToManyField(User, blank=True, related_name='allowed_themes')

    def getclass(self):
        return self.classname

    def __str__(self):
        return self.classname

    def __repr__(self):
        return self.classname

class PersonalInfo(models.Model):
    name = models.CharField(max_length=200, blank=True)
    atname = models.CharField(max_length=200, blank=True)
    slogan = models.CharField(max_length=200, blank=True)
    page = models.ForeignKey(Page, primary_key=True, on_delete=models.CASCADE, related_name='page')
    # avatar = models.FileField(upload_to='', blank=True, null=True)
    avatar = models.ImageField(upload_to='', blank=True, null=True)
    theme = models.ForeignKey(Themes, on_delete=models.CASCADE, blank=False, related_name='theme')

    def save(self, *args, **kwargs):
        super(PersonalInfo, self).save(*args, **kwargs)

        if bool(self.avatar):
            imag = Image.open(self.avatar.path)

            if imag.width > 500 or imag.height > 500:

                #bake image orienation
                orientation = 0
                for orient in ExifTags.TAGS.keys():
                    if ExifTags.TAGS[orient] == 'Orientation':
                        break
                exif = imag._getexif()
                if exif is not None:
                    if orient in exif: orientation = exif[orient]

                if orientation is 6: imag = imag.rotate(-90)
                elif orientation is 8: imag = imag.rotate(90)
                elif orientation is 3: imag = imag.rotate(180)
                elif orientation is 2: imag = imag.transpose(Image.FLIP_LEFT_RIGHT)
                elif orientation is 5: imag = imag.rotate(-90).transpose(Image.FLIP_LEFT_RIGHT)
                elif orientation is 7: imag = imag.rotate(90).transpose(Image.FLIP_LEFT_RIGHT)
                elif orientation is 4: imag = imag.rotate(180).transpose(Image.FLIP_LEFT_RIGHT)


                #crop image
                if imag.width < 500 and imag.height < 500: #smaller than max size
                    pass
                elif imag.height > imag.width: # make square
                    left = 0
                    right = imag.width
                    top = (imag.height - imag.width) / 2
                    bottom = (imag.height + imag.width) / 2
                    imag = imag.crop((left, top, right, bottom))
                elif imag.width > imag.height: # make square
                    left = (imag.width - imag.height) / 2
                    right = (imag.width + imag.height) / 2
                    top = 0
                    bottom = imag.height
                    imag = imag.crop((left, top, right, bottom))


                #resize image
                if imag.width > 500 or imag.height > 500:
                    output_size = (500, 500)
                    imag.thumbnail(output_size)


                imag.save(self.avatar.path)


class LinkType(models.Model):
    classname = models.CharField(max_length=10, blank=True)
    url = models.CharField(max_length=100, blank=True)
    placeholder = models.CharField(max_length=100, blank=True)
    description = models.CharField(max_length=400, blank=True)


    def getclass(self):
        return self.classname


class Links(models.Model):
    page = models.ForeignKey(Page, primary_key=False, on_delete=models.CASCADE, related_name='links')
    sort = models.IntegerField(blank=True)
    type = models.ForeignKey(LinkType, on_delete=models.CASCADE)
    value = models.CharField(max_length=200, blank=True)
    title = models.CharField(max_length=200, blank=True)
    external_widget_id = models.IntegerField(default=0)

    def save(self, *args, **kwargs):
        if self.sort is not None:
            return super(Links, self).save(*args, **kwargs)

        try:
            largest = Links.objects.filter().order_by('sort').last().sort
            self.sort = largest + 1
        except AttributeError:
            self.sort = 1

        return super(Links, self).save(*args, **kwargs)


    def has_type(self):
        return self.type_id is not None

    def getprefix(self):
        if self.type_id == 2 or self.type_id == 3 or self.type_id == 12 or self.type_id == 16 or self.type_id == 17 or self.type_id == 18:
            return "@{}".format(self.value)
        elif self.type_id == 1 and self.title == '':
            return 'YouTube'
        elif self.type_id == 4 and self.title == '':
            return 'Xing'
        elif self.type_id == 5 and self.title == '':
            return 'Apple Music'
        elif self.type_id == 7 and self.title == '':
            return 'Facebook'
        elif self.type_id == 8 and self.title == '':
            return 'SoundCloud'
        elif self.type_id == 9 and self.title == '':
            return 'Spotify'
        elif self.type_id == 11 and self.title == '':
            return 'Telephone number'
        elif self.type_id == 13 and self.title == '':
            return 'LinkedIn'
        elif self.type_id == 14 and self.title == '':
            return 'WhatsApp'
        elif self.type_id == 21 and self.title == '':
            return 'Review us on Google'
        elif self.type_id == 22 and self.title == '':
            return 'Business Card'
        else:
            return self.value

    def geturl(self):
        if self.type_id == 22:
            return "/pages/{}/{}/vcard".format(self.page_id, self.external_widget_id)
        else:
            return "{}{}".format(self.type.url, self.value)

class vcards(models.Model):
    vcard_title = models.CharField(max_length=200, blank=True)
    first_name = models.CharField(max_length=200, blank=True)
    last_name = models.CharField(max_length=200, blank=True)
    display_name = models.CharField(max_length=200, blank=True)
    private_website = models.CharField(max_length=200, blank=True)
    private_phonenumber = models.CharField(max_length=200, blank=True)
    private_street = models.CharField(max_length=200, blank=True)
    private_city = models.CharField(max_length=200, blank=True)
    private_plz = models.CharField(max_length=200, blank=True)
    private_country = models.CharField(max_length=200, blank=True)
    companyname = models.CharField(max_length=200, blank=True)
    job = models.CharField(max_length=200, blank=True)
    position = models.CharField(max_length=200, blank=True)
    email = models.CharField(max_length=200, blank=True)
    company_website = models.CharField(max_length=200, blank=True)
    company_phonenumber = models.CharField(max_length=200, blank=True)
    company_mobilenumber = models.CharField(max_length=200, blank=True)
    company_street = models.CharField(max_length=200, blank=True)
    company_city = models.CharField(max_length=200, blank=True)
    company_plz = models.CharField(max_length=200, blank=True)
    company_country = models.CharField(max_length=200, blank=True)